<?php 
/**
 * Database Creator
 *
 * @package uss.courier
 */
defined( 'ROOT_DIR' ) OR DIE;

$db = new stdClass();

/**
 * Courier Offices Or Branches
 *
 * Area and addresses where the courier company can be found
 */
$db->offices = "
	CREATE TABLE IF NOT EXISTS %{prefix}_offices (
		id INT NOT NULL PRIMARY KEY AUTO_INCREMENT,
		branch VARCHAR(255) NOT NULL,
		address VARCHAR(300) NOT NULL,
		state VARCHAR(255) NOT NULL,
		country VARCHAR(3),
		email VARCHAR(1000) COMMENT 'comma separated',
		phone VARCHAR(1000) COMMENT 'comma separated',
		status TINYINT NOT NULL DEFAULT 1 COMMENT 'enabled/disabled',
		period TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
		creator INT NOT NULL COMMENT 'user[id]',
		_default TINYINT NOT NULL DEFAULT 0
	);
";

/**
 * The staffs working in the courier company
 * Technically, the driver / shipper
 */
$db->staffs = "
	CREATE TABLE IF NOT EXISTS %{prefix}_staffs (
		id INT NOT NULL PRIMARY KEY AUTO_INCREMENT,
		userid INT NOT NULL,
		FOREIGN KEY(userid) REFERENCES %{prefix}_users(id) ON DELETE CASCADE,
		staff_key VARCHAR(20) NOT NULL UNIQUE,
		email VARCHAR(255),
		phone VARCHAR(20),
		status TINYINT NOT NULL DEFAULT 1 COMMENT 'active/inactive',
		period TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP
	);
";

/**
 * Shipment orders made by a customer
 *
 * ## STATUS
 *
 * - Queue
 * - Picked
 * - Received
 * - Shipped
 * - Delivered
 * - Rejected
 * - Request
 */
$db->orders = "
	CREATE TABLE IF NOT EXISTS %{prefix}_orders (
		id INT NOT NULL PRIMARY KEY AUTO_INCREMENT,
		name TINYTEXT,
		order_no VARCHAR(20) NOT NULL UNIQUE,
		office_id INT NOT NULL DEFAULT 0,
		status VARCHAR(100),
		period TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
		pickup_date VARCHAR(20),
		depature_date VARCHAR(20),
		staff_id INT,
		expected_delivery_date VARCHAR(20),
		order_comment TEXT,
		sender_id INT NOT NULL,
		receiver_id INT,
		price FLOAT DEFAULT 0.00,
		approval VARCHAR(20) DEFAULT 'pending'
	);
";

/* ----------------------------------------------------
 * ORDER META
 *
 * The "order_meta" table should be created with the pairs class:
 * The table data will contain
 * 
 * _ref = 'order_id' 
 * _key = from
 * _value = 'detail & address of sender'
 *
 * _ref = 'order_id'
 * _key = to
 * _ref = 'detail & address of receiver'
 *
 * ------------------------------------------------------
 */

/**
 * Item variation
 *
 * The type of item being shipped
 * Variation specifies category
 */

$db->variants = "
	CREATE TABLE IF NOT EXISTS %{prefix}_variants (
		id INT NOT NULL PRIMARY KEY AUTO_INCREMENT,
		name VARCHAR(255) COMMENT 'Package Group/Category'
	);
";

/**
 * Package in a single shipment
 *
 * This contains list of packages under a single order
 */
$db->packages = "
	CREATE TABLE IF NOT EXISTS %{prefix}_packages (
		id INT NOT NULL PRIMARY KEY AUTO_INCREMENT,
		order_id INT NOT NULL,
		FOREIGN KEY(order_id) REFERENCES %{prefix}_orders(id) ON DELETE CASCADE,
		item_no VARCHAR(10) NOT NULL UNIQUE,
		name VARCHAR(1000),
		weight FLOAT,
		quantity INT NOT NULL DEFAULT 1,
		var_id INT,
		image VARCHAR(1000)
	);
";

/**
 * Payment Methods
 *
 * The accepted payment method for the cargo company
 */
$db->gateways = "
	CREATE TABLE IF NOT EXISTS %{prefix}_gateways (
		id INT NOT NULL PRIMARY KEY AUTO_INCREMENT,
		name VARCHAR(255) NOT NULL,
		abbr VARCHAR(10),
		icon VARCHAR(255),
		model VARCHAR(100) COMMENT 'Bank,Crypto,3rd Party',
		detail TEXT COMMENT 'JSON',
		period TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
		status TINYINT NOT NULL DEFAULT 1 COMMENT '-1 To Hide'
	);
";

/**
 * Package in a single shipment
 *
 * The payments made by customers for a shipment
 */
$db->payments = "
	CREATE TABLE IF NOT EXISTS %{prefix}_payments (
		id INT NOT NULL PRIMARY KEY AUTO_INCREMENT,
		invoice_key VARCHAR(255) NOT NULL UNIQUE,
		order_id INT NULL,
		amount FLOAT NOT NULL DEFAULT 0,
		screenshot VARCHAR(1000),
		status VARCHAR(20) NOT NULL DEFAULT 'pending',
		method_id INT,
		method_detail TEXT,
		period TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
		payment_ref VARCHAR(100) NOT NULL UNIQUE
	);
";

/**
 * Shipment History
 */
$db->shipments = "
	CREATE TABLE IF NOT EXISTS %{prefix}_shipments (
		id INT NOT NULL PRIMARY KEY AUTO_INCREMENT,
		order_id INT NOT NULL,
		FOREIGN KEY(order_id) REFERENCES %{prefix}_orders(id) ON DELETE CASCADE,
		country VARCHAR(3) NOT NULL,
		state VARCHAR(255) NOT NULL,
		local_area VARCHAR(1000),
		longitude VARCHAR(20),
		latitude VARCHAR(20),
		period TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP
	);
";

/**
 * ESTABLISH DATABASE TABLE
 */
foreach( $db as $SQL ) {
	
	/**
	 * Validate SQL Query
	 */
	$SQL = Core::replace_var( $SQL, array( 'prefix' => DB_TABLE_PREFIX ) );
	
	/**
	 * Execute The SQL Data
	 */
	$result = Uss::$global['mysqli']->query( $SQL );
	
	/**
	 * Print Error!
	 */
	if( !$result ) {
		
		$Error = "SQL SYNTAX ERROR: \n {$SQL} \n\t" . Uss::$global['mysqli']->error . "\n\n";
		
		/** 
		 * Throw Exception
		 */
		throw new Exception( $Error );
		
	}
	
};

/**
 * Meta
 */
self::$orderMeta = new pairs( Uss::$global['mysqli'], DB_TABLE_PREFIX . '_order_meta' );

/**
 * Link parent & add constraint
 */
self::$orderMeta->linkParentTable( DB_TABLE_PREFIX . "_orders", DB_TABLE_PREFIX . "_order_id" );

